function cds_portfolio = import_cds_bloomberg(file_name, business_days)
% --------------------------------------------------------------------------------------------------
% Import the historical CDS data in the Bloomberg format.
% --------------------------------------------------------------------------------------------------
% file_name         ... Excel file name with CDS history
% business_days     ... cell array of business days
% --------------------------------------------------------------------------------------------------
% sample call: import_cds_bloomberg('c:/thesis/data/bloomberg/cds series5.xls', swap_dates)
% --------------------------------------------------------------------------------------------------

% Import data
[num, txt] = xlsread(file_name, 1);
[num2, txt2] = xlsread(file_name, 2);
[num3, txt3] = xlsread(file_name, 3);

% Define relevant columns
used_cols = sort([1:5:size(num,2) 2:5:size(num,2) 3:5:size(num,2)]);
used_cols2 = sort([1:5:size(num2,2) 2:5:size(num2,2) 3:5:size(num2,2)]);
used_cols3 = sort([1:5:size(num3,2) 2:5:size(num3,2) 3:5:size(num3,2)]);
date_cols = 1:5:size(num,2);
date_cols2 = 1:5:size(num2,2);
date_cols3 = 1:5:size(num3,2);

% Merge data price data
n1 = length(used_cols);
n2 = n1 + length(used_cols2);
n3 = n2 + length(used_cols3);
prices = zeros( max([size(num,1) size(num2,1) size(num3,1)]), n3);
prices(1:size(num,1), 1:n1) = num(:, used_cols);
prices(1:size(num2,1), (n1+1):n2) = num2(:, used_cols2);
prices(1:size(num3,1), (n2+1):n3) = num3(:, used_cols3);

% Clean date data and merge ticker data
n1 = length(date_cols);
n2 = n1 + length(date_cols2);
n3 = n2 + length(date_cols3);
dates = txt(3:end, date_cols);
dates2 = txt2(3:end, date_cols2);
dates3 = txt3(3:end, date_cols3);
tickers = [txt(1,date_cols) txt2(1,date_cols2) txt3(1,date_cols3)];

% Convert merged data into portfolio of CDS
cds_portfolio = [];
for i=1:n3
    % Extract relevant date column
    disp(i);
    if (i <= n1)
        cds_dates = dates(:,i);
    elseif (i <= n2)
        cds_dates = dates2(:,i-n1);
    else
        cds_dates = dates3(:,i-n2);
    end
    
    % Remove empty entries and convert date into format MM/DD/YYYY
    [trash, empty] = ismember(cds_dates, '');
    cds_datenum = datenum(cds_dates(~empty));
    cds_dates = {};
    for j=1:length(cds_datenum)
        cds_dates{j,1} = datestr(cds_datenum(j), 23);
    end
    
    % Delete non-business days
    available = find(ismember(cds_dates, business_days));
    cds_dates = {cds_dates{available}}';
    market_price = prices(available,i*3);
    bid_ask_spread = prices(available,i*3-1)-prices(available,i*3-2);
    
    % Replace missing values/zeros in bid-ask spread with last available data
    problems = find(isnan(bid_ask_spread) + (bid_ask_spread<=0));
    for j=1:length(problems)
        if (problems(j)==j)
            fix = find((problems' == 1:length(problems)), 1, 'last') + 1;
        else
            fix = 1;
            for k=(problems(j)-1):(-1):1
                if (~ismember(k, problems))
                    fix = max(fix, k);
                end
            end
        end
        bid_ask_spread(problems(j)) = bid_ask_spread(fix);
    end    
    
    % Determine matching IMM dates
    imm_dates = match_imm_dates(datenum(cds_dates), 5);
    
    % Calculate time to maturity (in years)
    T = zeros(length(cds_dates),1);
    for j=1:length(cds_dates)
        T(j) = date_diff_years(datenum(cds_dates(j)), imm_dates(j));
    end
    
    % Put everything into a structure
    cds = struct('company', tickers{i}, 'dates', {{datenum(cds_dates)}}, 'market_price', {{market_price}}, ...
                 'T', {{T}}, 'bid_ask_spread', {{bid_ask_spread}});
    if (i == 1)
        cds_portfolio = cds;
    else
        cds_portfolio(i) = cds;
    end
end

